import { MenuItemOptions } from './menu';
import { Subject, Observable } from 'rxjs';
export interface ClipboardContent {
    text: string;
    html?: string;
}
export interface MessageBoxOptions {
    type: 'warning' | 'error';
    message: string;
    detail?: string;
    buttons: string[];
    defaultId?: number;
    cancelId?: number;
}
export interface MessageBoxResult {
    response: number;
}
export declare abstract class FileTransfer {
    abstract getName(): string;
    abstract getSize(): number;
    abstract close(): void;
    getSpeed(): number;
    getCompletedBytes(): number;
    getStatus(): string;
    getTotalSize(): number;
    isComplete(): boolean;
    isCancelled(): boolean;
    cancel(): void;
    setStatus(status: string): void;
    setTotalSize(size: number): void;
    setCompleted(completed: boolean): void;
    protected increaseProgress(bytes: number): void;
    private completedBytes;
    private totalSize;
    private lastChunkStartTime;
    private lastChunkSpeed;
    private cancelled;
    private completed;
    private status;
}
export declare abstract class FileDownload extends FileTransfer {
    abstract write(buffer: Uint8Array): Promise<void>;
}
export declare abstract class DirectoryDownload extends FileTransfer {
    abstract createDirectory(relativePath: string): Promise<void>;
    abstract createFile(relativePath: string, mode: number, size: number): Promise<FileDownload>;
}
export declare abstract class FileUpload extends FileTransfer {
    abstract getMode(): number;
    abstract read(): Promise<Uint8Array>;
    readAll(): Promise<Uint8Array>;
}
export interface FileUploadOptions {
    multiple: boolean;
}
export declare class DirectoryUpload {
    private name;
    private childrens;
    constructor(name?: string);
    getName(): string;
    getChildrens(): (FileUpload | DirectoryUpload)[];
    pushChildren(item: FileUpload | DirectoryUpload): void;
}
export type PlatformTheme = 'light' | 'dark';
export declare abstract class PlatformService {
    supportsWindowControls: boolean;
    get fileTransferStarted$(): Observable<FileTransfer>;
    get displayMetricsChanged$(): Observable<void>;
    get themeChanged$(): Observable<PlatformTheme>;
    protected fileTransferStarted: Subject<FileTransfer>;
    protected displayMetricsChanged: Subject<void>;
    protected themeChanged: Subject<PlatformTheme>;
    abstract readClipboard(): string;
    abstract setClipboard(content: ClipboardContent): void;
    abstract loadConfig(): Promise<string>;
    abstract saveConfig(content: string): Promise<void>;
    abstract startDownload(name: string, mode: number, size: number): Promise<FileDownload | null>;
    abstract startDownloadDirectory(name: string, estimatedSize?: number): Promise<DirectoryDownload | null>;
    abstract startUpload(options?: FileUploadOptions): Promise<FileUpload[]>;
    abstract startUploadDirectory(paths?: string[]): Promise<DirectoryUpload>;
    startUploadFromDragEvent(event: DragEvent, multiple?: boolean): Promise<DirectoryUpload>;
    getConfigPath(): string | null;
    showItemInFolder(path: string): void;
    isProcessRunning(name: string): Promise<boolean>;
    installPlugin(name: string, version: string): Promise<void>;
    uninstallPlugin(name: string): Promise<void>;
    getWinSCPPath(): string | null;
    exec(app: string, argv: string[]): Promise<void>;
    isShellIntegrationSupported(): boolean;
    isShellIntegrationInstalled(): Promise<boolean>;
    installShellIntegration(): Promise<void>;
    uninstallShellIntegration(): Promise<void>;
    openPath(path: string): void;
    getTheme(): PlatformTheme;
    abstract getOSRelease(): string;
    abstract getAppVersion(): string;
    abstract openExternal(url: string): void;
    abstract listFonts(): Promise<string[]>;
    abstract setErrorHandler(handler: (_: any) => void): void;
    abstract popupContextMenu(menu: MenuItemOptions[], event?: MouseEvent): void;
    abstract showMessageBox(options: MessageBoxOptions): Promise<MessageBoxResult>;
    abstract pickDirectory(): Promise<string | null>;
    abstract quit(): void;
}
export declare class HTMLFileUpload extends FileUpload {
    private file;
    private stream;
    private reader;
    constructor(file: File);
    getName(): string;
    getMode(): number;
    getSize(): number;
    read(): Promise<Uint8Array>;
    bringToFront(): void;
    close(): void;
}
